<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\Resume;
use App\Models\Experience;
use App\Models\Education;
use App\Models\Certification;
use App\Models\Skill;
use App\Models\ExperiencePoint;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Auth;
use Barryvdh\DomPDF\Facade\Pdf;
use App\Services\LLMService;


class ResumeController extends Controller
{
    // STEP 1: Personal
    public function personal()
    {
        $resume = Resume::firstOrCreate(['user_id' => Auth::id()]);
        return view('resumes.personal', compact('resume'));
    }

    public function savePersonal(Request $request)
    {
        $resume = Resume::where('user_id', Auth::id())->firstOrFail();

        $resume->update($request->only([
            'first_name','last_name','job_title','email','phone','address',
            'city','state','country','postal_code','linkedin_url','portfolio_url'
        ]));

        return redirect()->route('resume.summary', $resume->id)
                         ->with('success', 'Personal details saved!');
    }

    // STEP 2: Summary
    public function summary(Resume $resume)
    {
        return view('resumes.summary', compact('resume'));
    }

    public function saveSummary(Request $request, Resume $resume)
    {
        $resume->update(['summary' => $request->summary]);
        return redirect()->route('resume.experience', $resume->id)
                         ->with('success', 'Summary saved!');
    }

    // STEP 3: Experience
    public function experience(Resume $resume)
    {
         //$resume->load('experiences.points');
        return view('resumes.experience', compact('resume'));
    }

    public function saveExperience(Request $request, Resume $resume)
    {
         $existingIds = $resume->experiences()->pluck('id')->toArray();
    $submittedIds = collect($request->experience)->pluck('id')->filter()->toArray();

    // Delete removed experiences
    $resume->experiences()->whereNotIn('id', $submittedIds)->delete();

    foreach ($request->experience as $expData) {
        if (empty($expData['company']) && empty($expData['position'])) continue;

        // --- Create or update experience ---
        if (!empty($expData['id'])) {
            $experience = $resume->experiences()->find($expData['id']);
            if ($experience) {
                $experience->update([
                    'company'     => $expData['company'],
                    'position'    => $expData['position'] ?? null,
                    'location'    => $expData['location'] ?? null,
                    'start_date'  => $expData['start_date'] ?? null,
                    'end_date'    => $expData['end_date'] ?? null,
                ]);
            }
        } else {
            $experience = $resume->experiences()->create([
                'company'     => $expData['company'],
                'position'    => $expData['position'] ?? null,
                'location'    => $expData['location'] ?? null,
                'start_date'  => $expData['start_date'] ?? null,
                'end_date'    => $expData['end_date'] ?? null,
            ]);
        }

        // --- Handle Experience Points ---
        if ($experience && isset($expData['points'])) {
            $experience->points()->delete(); // remove old ones
            foreach ($expData['points'] as $point) {
                if (!empty($point)) {
                    $experience->points()->create(['description' => trim($point)]);
                }
            }
        }
    }

    return redirect()->route('resume.education', $resume->id)
                     ->with('success', 'Experience & points saved successfully!');
    }

    // STEP 4: Education
    public function education(Resume $resume)
    {
        return view('resumes.education', compact('resume'));
    }

    public function saveEducation(Request $request, Resume $resume)
    {
        $existingIds = $resume->educations()->pluck('id')->toArray();
        $submittedIds = collect($request->education)->pluck('id')->filter()->toArray();

        $resume->educations()->whereNotIn('id', $submittedIds)->delete();

        foreach ($request->education as $eduData) {
            if (empty($eduData['institution'])) continue;

            if (!empty($eduData['id'])) {
                $education = $resume->educations()->find($eduData['id']);
                if ($education) {
                    $education->update([
    'institution'    => $eduData['institution'],
    'degree'         => $eduData['degree'] ?? null,
    'field_of_study' => $eduData['field_of_study'] ?? null,
    'start_year'     => $eduData['start_year'] ?? null,
    'end_year'       => $eduData['end_year'] ?? null,
    'description'    => $eduData['description'] ?? null,
]);
                }
            } else {
                $resume->educations()->create([
                    'institution'    => $eduData['institution'],
    'degree'         => $eduData['degree'] ?? null,
    'field_of_study' => $eduData['field_of_study'] ?? null,
    'start_year'     => $eduData['start_year'] ?? null,
    'end_year'       => $eduData['end_year'] ?? null,
    'description'    => $eduData['description'] ?? null,
                ]);
            }
        }

        return redirect()->route('resume.certifications', $resume->id)
                         ->with('success', 'Education saved successfully!');
    }

    // STEP 5: Certifications / Skills
    public function certifications(Resume $resume)
    {
        return view('resumes.certifications', compact('resume'));
    }

    public function saveCertifications(Request $request, Resume $resume)
    {
        $existingIds = $resume->certifications()->pluck('id')->toArray();
        $submittedIds = collect($request->certifications)->pluck('id')->filter()->toArray();

        $resume->certifications()->whereNotIn('id', $submittedIds)->delete();

        foreach ($request->certifications as $certData) {
            if (empty($certData['title'])) continue;

            if (!empty($certData['id'])) {
                $cert = $resume->certifications()->find($certData['id']);
                if ($cert) {
                    $cert->update([
                        'title'       => $certData['title'],
                        'issuer'      => $certData['issuer'] ?? null,
                        'issue_date'  => $certData['issue_date'] ?? null,
                        'expiry_date' => $certData['expiry_date'] ?? null,
                    ]);
                }
            } else {
                $resume->certifications()->create([
                    'title'       => $certData['title'],
                    'issuer'      => $certData['issuer'] ?? null,
                    'issue_date'  => $certData['issue_date'] ?? null,
                    'expiry_date' => $certData['expiry_date'] ?? null,
                ]);
            }
        }

        // Save Skills (overwrite each time)
        $resume->skills()->delete();
        if ($request->filled('skills')) {
            foreach (explode(',', $request->skills) as $skill) {
                $resume->skills()->create(['skill_name' => trim($skill)]);
            }
        }

        return redirect()->route('resume.personal')
                         ->with('success', 'All resume sections saved successfully!');
    }


    public function preview(Resume $resume, $template = 'classic')
{
    $resume->load(['experiences.points', 'educations', 'certifications', 'skills']);
    return view("resumes.templates.$template", compact('resume'));
}
public function download(Resume $resume)
{
    $resume->load(['experiences.points', 'educations', 'certifications', 'skills']);
    $pdf = Pdf::loadView('resumes.templates.classic', compact('resume'));
    return $pdf->download('resume.pdf');
}


public function generateSummary(Resume $resume)
{
    $prompt = "
    Write a professional summary (3-5 sentences) based on this resume data:
    Name: {$resume->first_name} {$resume->last_name}
    Title: {$resume->job_title}
    Experience: " . $resume->experiences->pluck('position')->implode(', ') . "
    Skills: " . $resume->skills->pluck('skill_name')->implode(', ') . "
    The tone should be professional, concise, and ATS-friendly.
    ";

    $response = LLMService::analyze($prompt, 'gemini');
    $text = $response['choices'][0]['message']['content'] ?? '';

    $resume->update(['summary' => trim(strip_tags($text))]);

    return back()->with('success', 'AI-generated professional summary added!');
}

public function generateExperience(Resume $resume, Experience $experience)
{
    $prompt = "
    Generate 4-6 bullet points describing quantifiable achievements and responsibilities for the following role:
    Job Title: {$experience->position}
    Company: {$experience->company}
    Skills: " . $resume->skills->pluck('skill_name')->implode(', ') . "
    Output as JSON array like: [\"bullet 1\", \"bullet 2\"]
    ";

    $response = LLMService::analyze($prompt, 'openai');
    $content = $response['choices'][0]['message']['content'] ?? '[]';

    // Extract bullets safely
    $bullets = json_decode($content, true);
    if (!$bullets && preg_match('/\[(.*?)\]/s', $content, $m)) {
        $bullets = json_decode("[" . $m[1] . "]", true);
    }

    if (is_array($bullets)) {
        $experience->points()->delete();
        foreach ($bullets as $line) {
            $experience->points()->create(['description' => trim($line)]);
        }
    }

    return back()->with('success', 'AI-generated experience bullets added!');
}


public function generateSkills(Resume $resume)
{
    $prompt = "
    Given this resume information, generate an optimized skill list for ATS systems.
    Include both technical and soft skills, comma-separated.
    Resume summary: {$resume->summary}
    Experience roles: " . $resume->experiences->pluck('position')->implode(', ') . "
    Skills: " . $resume->skills->pluck('skill_name')->implode(', ') . "
    ";

    $response = LLMService::analyze($prompt, 'gemini');
    $content = $response['choices'][0]['message']['content'] ?? '';

    // Extract comma-separated list
    $skills = array_filter(array_map('trim', explode(',', strip_tags($content))));

    if ($skills) {
        $resume->skills()->delete();
        foreach ($skills as $skill) {
            $resume->skills()->create(['skill_name' => ucfirst($skill)]);
        }
    }

    return back()->with('success', 'AI-enhanced skills generated!');
}
}
